/*
   This program demonstrates that I/O buffering is
   essential for good I/O performance, the size of
   the buffer matters, and that "double buffering"
   is far better than no buffering, but it may not
   be optimal.

   This program can run in two different modes.

   In one mode, RAW=true, this program reads one
   byte at a time from System.in using a buffer
   whose size is determined by the constant BUFFER_SIZE.

   In the other mode, RAW=false, there is a Scanner
   object wrapped around System.in which provides
   another buffer (so we have "double buffering").

   The BUFFER_SIZE parameter lets you set the size of
   the buffer used in the BufferedInputStream object
   that System.in is a reference to.

   Use I/O redirection at the command prompt to run a
   large file through this program and get a sense of
   how buffer size effects performance time.

   C:\>java Ex0_UsingAnInputBuffer < big.txt

   The worst performance should be when RAW=true and
   BUFFER_SIZE=1. In this case there is essentially no
   buffering. Every call to System.in.read() must access
   the file through the operating system

   If you increase BUFFER_SIZE from 5 to 10 to 100, you
   should see significant improvements in performance.

   If you set BUFFER_SIZE to 1000 (or more) you should
   get optimal performance.

   On the other hand, if RAW=false, then the value of
   BUFFER_SIZE no longer seems to matter (try putting
   it back to 1). The buffer in Scanner makes the
   performance pretty good (but not optimal).

   This shows that buffering is essential. The size of
   the buffer matters (up to a point) and having two
   buffers need not be better than one buffer, but two
   buffers is not likely to be a performance disaster.
   (Sometimes, in real applications, it can be difficult
   to avoid having more than one buffer.)
*/
import java.util.Scanner;
import java.io.*;

public class Ex0_UsingAnInputBuffer
{
   final static boolean RAW = false;

   public static void main(String[] args) throws IOException
   {
      final int BUFFER_SIZE = 1000;  // Try values up to 100,000.
      System.setIn( new BufferedInputStream( new FileInputStream(FileDescriptor.in), BUFFER_SIZE ) );

      Scanner scanner = new Scanner( System.in );

      long startTime = System.currentTimeMillis(); // "start" the timer

      if (RAW)  // don't use the scanner
      {
         int b = System.in.read();
         while ( -1 != b )
         {
            b = System.in.read();
         }
      }
      else  // use the scanner
      {
         while ( scanner.hasNextLine() )
         {
            String oneLine = scanner.nextLine();
         }
      }

      long endTime = System.currentTimeMillis();  // "stop" the "timer

      System.err.println("Total execution time (wct): " + (endTime - startTime) );
   }
}